// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_BLUETOOTH_ENDPOINT_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_BLUETOOTH_ENDPOINT_H

#include "common.h"
#include "IBluetoothCallbacks.h"
#include "ProtocolEndpointBase.h"

#include <string>

/**
 * This class implements an endpoint that handles Bluetooth pairing requests
 * over the GAL protocol. The head unit must also subclass IBluetoothCallbacks
 * and implement all the methods so that the GAL receiver can interface with
 * the head unit hardware. The call sequence would look something like this:
 * <br>
 * <pre>
 *      galReceiver->init();
 *      ... Initialization code ...
 *      BluetoothEndpoint* endpoint = new BluetoothEndpoint(serviceId,
 *              galReceiver->messageRouter());
 *      endpoint->registerCallbacks(callbacks); // Subclassed from IBluetoothCallbacks.
 *      endpoint->registerCarBluetoothAddress(...);
 *      endpoint->registerPairingMethod(...);
 *      endpoint->registerPairingMethod(...); // Multiple methods are allowed.
 *      galReceiver->registerService(endpoint);
 *      ... Other Initialization code ...
 *      galReceiver->start();
 *      ... When calbacks->onPairingRequest() is called, the head unit should
 *          make it prepared for pairing. And call onReadForPairing. ...
 *      endpoint->onReadyForPairing();
 *      ... After Bluetooth pairing is initiated by the phone, send authentication data.
 *      endpoint->sendAuthData(authData);
 * </pre>
 */
class BluetoothEndpoint : public ProtocolEndpointBase {
public:
    BluetoothEndpoint(uint8_t id, MessageRouter *router)
            : ProtocolEndpointBase(id, router, false) {}

    /**
     * Registers the Bluetooth address of the car with the GAL receiver. This
     * must be done during intialization as described in the high level
     * description of this class.
     *
     * @param carBluetoothAddress The bluetooth address of the car. For example,
     *        "00:11:22:AA:BB:CC".
     */
    void registerCarBluetoothAddress(const string& carBluetoothAddress);

    /**
     * Registers a Bluetooth pairing method with the GAL receiver. The head unit
     * should register all the pairing methods it supports, by calling this
     * method repeatedly during initialization.
     * @param pairingMethod Can be BLUETOOTH_PAIRING_NUMERIC_COMPARISON or BLUETOOTH_PAIRING_PIN.
     */
    void registerPairingMethod(const BluetoothPairingMethod& pairingMethod);

    void registerCallbacks(const shared_ptr<IBluetoothCallbacks>& callbacks);

    /**
     * Call this if the head unit cannot be prepared for pairing within 1 second
     * after |IBluetoothCallbacks::onPairingRequest| is called.
     * Then the phone will assume that the head unit will call
     * |onReadyForPairing| after a while.
     */
    void requestDelayedPairing();

    /**
     * After |IBluetoothCallbacks::onPairingRequest| is called, the head unit
     * must make itself ready to be paired with the phone through Bluetooth,
     * even if the head unit has already been paired with the phone. And when
     * the headunit becomes ready for pairing, it must call |onReadyForPairing|.
     *
     * @param alreadyPaired |true| if and only if the head unit already has link
     *         key (pairing information) for the phone (In other words, if the
     *         head unit thinks it's paired with the phone).
     */
    void onReadyForPairing(bool alreadyPaired);

    /**
     * Sends a Bluetooth authentication data (numeric passkey or PIN) to the
     * phone. The phone will use these authentication data to pair itself with
     * the car through Bluetooth.
     *
     * Call this function when the head unit is ready for pairing and it has
     * obtained the authentication data from its Bluetooth stack.
     *
     * If the phone and the car are not paired through Bluetooth yet, the head
     * unit must call this function, even if authentication data are actually
     * not required. If the head unit doesn't call this function, Bluetooth
     * pairing will not be done.
     *
     * @param authData The authentication data (For example, a numeric passkey
     *        or a PIN).
     */
    void sendAuthData(const string& authData);

    // ProtocolEndpointBase implementation.

    void addDiscoveryInfo(ServiceDiscoveryResponse* sdr);

    int routeMessage(uint8_t channelId, uint16_t type,
                     const shared_ptr<IoBuffer>& msg);

private:
    void handlePairingRequest(const BluetoothPairingRequest& req);
    /**
     * @param status Can be |STATUS_SUCCESS| or
     *         |STATUS_BLUETOOTH_PAIRING_DELAYED|.
     * @param alreadyPaired Whether the head unit has link key (pairing
     *         information). Ignored if |status| is not |STATUS_SUCCESS|.
     */
    void sendPairingResponse(int32_t status, bool alreadyPaired);

    string mCarBluetoothAddress;
    set<BluetoothPairingMethod> mSupportedPairingMethods;
    shared_ptr<IBluetoothCallbacks> mCallbacks;
};

#endif // ANDROID_AUTO_PROJECTION_PROTOCOL_BLUETOOTH_ENDPOINT_H
